<?php

namespace Senh\Lib\Bonuses;

/**
 * Class AbstractBonuses
 * @package Senh\Lib\Bonuses
 * Checks membership by following priority:
 * Get parameter, Session, Cookie
 * The parameter value will automatically be stored in the session and as a cookie
 *
 * Why use Session? Sometimes a specific flow (index to checkout) demands a membership include or exclude
 */
abstract class AbstractBonuses implements BonusesInterface
{

    const COOKIE_NAME = 'mb';
    const SESSION_NAME = 'mb';

    /**
     * @var bool
     */
    protected $showMembership;

    /**
     * AbstractBonuses constructor.
     */
    public function __construct()
    {
        $this->showMembership = true;

        if ($this->hasValidMembershipParameters()) {
            $this->showMembership = $this->showMembershipByParameters();
            $this->setCookie($this->showMembership);
            $this->setSession($this->showMembership);

            return;
        }

        if ($this->hasCookie()) {
            $this->showMembership = $this->getCookie();
        }

        if ($this->hasSession()) {
            $this->showMembership = $this->getSession();
        }
    }

    /**
     * @return int
     */
    public function getBonusCount()
    {
        if (!$this->showMembership()) {
            return $this->getBonusCountWithoutMembership();
        }

        return $this->getDefaultBonusCount();
    }

    public function getBonusCountWithoutMembership()
    {
        return $this->getDefaultBonusCount() - 1;
    }

    /**
     * @return bool
     */
    public function showMembership()
    {
        return $this->showMembership;
    }

    /**
     * @param bool $showMembership
     * @param bool $overrideSession
     * @param bool $overrideCookie
     */
    public function setShowMembership($showMembership, $overrideSession, $overrideCookie)
    {
        $this->showMembership = $showMembership;

        if ($overrideSession) {
            $this->setSession($showMembership);
        }

        if ($overrideCookie) {
            $this->setCookie($showMembership);
        }
    }

    /**
     * @return bool
     */
    protected function hasValidMembershipParameters()
    {
        return isset($_GET['mb']) && (strtolower($_GET['mb']) === 'ex' || strtolower($_GET['mb']) === 'in');
    }

    /**
     * @return bool
     */
    protected function showMembershipByParameters()
    {
        if (!$this->hasValidMembershipParameters()) {
            return true;
        }

        return $_GET['mb'] === 'in';
    }


    /**
     * @return string
     */
    protected function getSessionName()
    {
        return self::SESSION_NAME;
    }

    /**
     * @return bool
     */
    protected function hasSession()
    {
        $sessionName = $this->getSessionName();

        return isset($_SESSION[$sessionName]);
    }

    /**
     * @return bool|null
     */
    protected function getSession()
    {
        if (!$this->hasSession()) {
            return null;
        }

        $sessionName = $this->getSessionName();

        return $_SESSION[$sessionName];
    }

    /**
     * @param bool $showMembership
     */
    protected function setSession($showMembership)
    {
        $sessionName = $this->getSessionName();
        $_SESSION[$sessionName] = $showMembership;
    }
    
    /**
     * @return string
     */
    protected function getCookieName()
    {
        return self::COOKIE_NAME;
    }

    /**
     * @return bool
     */
    protected function hasCookie()
    {
        $cookieName = $this->getCookieName();

        return isset($_COOKIE[$cookieName]);
    }

    /**
     * @return bool|null
     */
    protected function getCookie()
    {
        if (!$this->hasCookie()) {
            return null;
        }

        $cookieName = $this->getCookieName();

        return (bool)$_COOKIE[$cookieName];
    }

    /**
     * @param bool $showMembership
     */
    protected function setCookie($showMembership)
    {
        $showMembership = $showMembership ? '1' : '0';
        $cookieName = $this->getCookieName();
        setcookie($cookieName, $showMembership, 0, '/');
    }
}