<?php

/**
 * System Pages Methods
 *
 * @package WishListMember
 */

namespace WishListMember;

/**
* System Pages Methods trait
*/
trait System_Pages_Methods
{
    /**
     * Retrieves the WishList Member page.
     * It also generates the page if it does not exist
     *
     * @param  boolean $link TRUE to return the link
     *                      or FALSE to just return the page ID.
     * @return string|integer   URL | Page ID
     */
    public function magic_page($link = true)
    {
        /*
         * This method has been totally recoded
         * to make it work with the new WP 2.9
         * Trash feature.
         *
         * It also takes care of the bug wherein
         * WishList Member generates hundreds to
         * thousands of "WishList Member" magic
         * pages when conflicting plugins are
         * installed.
         */
        global $wpdb;

        /*
         * Lets use static variable to prevent multiple DB calls
         * to get the magic page ID.
         */
        static $wpm_page = null;
        if ($wpm_page) {
            if ($link) {
                return get_permalink($wpm_page);
            } else {
                return $wpm_page;
            }
        }

        $save    = false;
        $date    = '2000-01-01 00:00:00'; // This will act as our "marker".
        $title   = 'WishList Member';

        // Get what's stored and make sure it is not cached.
        $wpm_page = $this->get_option('magic_page', false, true);
        if ($wpm_page) {
            $page_data = (object) get_post($wpm_page);
        }

        if (empty($page_data) || empty(get_object_vars($page_data))) {
            $wpm_page = false;
        }

        // If what's stored is not a magic page then.
        // Let's try going straight to the db.
        if (! $wpm_page || 'publish' !== $page_data->post_status || 'page' !== $page_data->post_type || $date !== $page_data->post_date) {
            $wpm_page = $this->get_magic_page_by_marker($date);
            $save     = true;
        }
        // Still not found? let's do some magic.
        if (! $wpm_page) {
            $to_insert = [
                'post_title'     => $title,
                'post_content'   => '',
                'post_status'    => 'publish',
                'post_author'    => '1',
                'ping_status'    => 'closed',
                'comment_status' => 'closed',
                'post_type'      => 'page',
                'post_date'      => $date,
            ];
            $wpm_page  = wp_insert_post($to_insert);

            if (! $wpm_page) {
                /*
                 * something bad happened. our post was not added somehow. it should
                 * be added in normal situations and the only reason why an apparent
                 * failure appears is due to conflicting plugins.
                 *
                 * some conflicting plugins are installed mess up with the return
                 * value of wp_insert_post which evaluates to false
                 *
                 * our solution is to attempt to get the magic page's ID by searching
                 * for it straight in the database.  obviously, this is just a
                 * workaround and may not work sometimes.
                 *
                 * our database search criteria are as follows:
                 * - post_date = 2000-01-01 00:00:00 (marker)
                 * - post_status = publish
                 * - post_type = page
                 */
                $wpm_page = $this->get_magic_page_by_marker($date);
            }
            $save = true;
        }

        if ($save) {
            $this->save_option('magic_page', $wpm_page);
        }

        if ($link) {
            return get_permalink($wpm_page);
        } else {
            return $wpm_page;
        }
    }

    /**
     * Retrieve WishList Member magic page by date marker
     *
     * @param string $date_marker Data marker for WishList Member magic page
     * @return integer Page ID of WishList Member magic page
     */
    private function get_magic_page_by_marker($date_marker)
    {
        global $wpdb;
        return (int) $wpdb->get_var(
            $wpdb->prepare(
                'SELECT `ID` FROM `' . $wpdb->posts . '` WHERE `post_date`=%s AND `post_status`="publish" AND `post_status`<>%s AND `post_type`="page" ORDER BY `ID` DESC LIMIT 1',
                $date_marker,
                microtime(true) // Make query unique to prevent cached results.
            )
        );
    }
    /**
     * Get Unsubscribe Confirmation URL
     *
     * @return string
     */
    public function unsubscribe_url()
    {
        $url = $this->get_option('unsubscribe_internal');
        $url = $url ? get_permalink($url) : $this->get_option('unsubscribe');
        return $url;
    }

    /**
     * Get Unsubscribe Confirmation URL
     *
     * @return string
     */
    public function resubscribe_url()
    {
        $url = $this->get_option('resubscribe_internal');
        $url = $url ? get_permalink($url) : $this->get_option('resubscribe');
        return $url;
    }

    /**
     * Get Redirect URL
     *
     * @param  string $global_type     Global type.
     * @param  string $global_page     Global page.
     * @param  string $global_url      Global url.
     * @param  string $global_message  Global message.
     * @param  string $perpage_type    Per page type.
     * @param  string $perpage_page    Per page page.
     * @param  string $perpage_url     Per page url.
     * @param  string $perpage_message Per page message.
     * @param  string $default         Default value to return if no URL is found.
     * @return string                  Redirect URL
     */
    public function get_redirect_url($global_type, $global_page, $global_url, $global_message, $perpage_type, $perpage_page, $perpage_url, $perpage_message, $default = '')
    {
        global $post;

        $post_id = (int) apply_filters('wishlistmember_redirect_url_post_id', (int) $post->ID, $perpage_type);

        $url = '';
        if (! $default) {
            $default = get_bloginfo('url');
        }

        // Per page/post setting.
        if ($post_id > 0) {
            switch ($this->get_option($perpage_type . $post_id)) {
                case 'message':
                    $url = add_query_arg(
                        [
                            'sp'  => $perpage_message,
                            'pid' => $post_id,
                        ],
                        $this->magic_page()
                    );
                    break;
                case 'page':
                    $url = get_permalink($this->get_option($perpage_page . $post_id));
                    break;
                case 'url':
                    $url = $this->get_option($perpage_url . $post_id);
                    break;
                default:
                    $url = '';
            }
        }

        // Global setting.
        if (! $url) {
            $type = $this->get_option($global_type);
            if (false === $type) {
                // V2.9 and below.
                $url = $this->get_option($global_page);
                $url = $url ? get_permalink($url) : $this->get_option($global_url);
            } else {
                switch ($type) {
                    case 'text':
                        $url = add_query_arg(['sp' => $global_message], $this->magic_page());
                        break;
                    case 'internal':
                        $url = get_permalink($this->get_option($global_page));
                        break;
                    default:
                        $url = $this->get_option($global_url);
                }
            }
        }

        $url = wlm_trim($url);
        // Default to global message if $url is still empty.
        if (! $url) {
            $url = add_query_arg(
                [
                    'sp'  => $perpage_message,
                    'pid' => $post_id,
                ],
                $this->magic_page()
            );
        }

        // If still no URL then default to homepage.
        $url = ! $url ? $default : $this->append_current_uri($url);
        return $url;
    }

    /**
     * Get Non-Members URL
     *
     * @param  string $default Optional default value.
     * @return string
     */
    public function non_members_url($default = '')
    {
        return $this->get_redirect_url(
            'non_members_error_page_type',
            'non_members_error_page_internal',
            'non_members_error_page',
            'non_members_error_page',
            'non_members_error_page_type_',
            'non_members_error_page_internal_',
            'non_members_error_page_',
            'non_members_error_page',
            $default
        );
    }

    /**
     * Get Wrong Level URL
     *
     * @param  string $default Optional default value.
     * @return string
     */
    public function wrong_level_url($default = '')
    {
        return $this->get_redirect_url(
            'wrong_level_error_page_type',
            'wrong_level_error_page_internal',
            'wrong_level_error_page',
            'wrong_level_error_page',
            'wrong_level_error_page_type_',
            'wrong_level_error_page_internal_',
            'wrong_level_error_page_',
            'wrong_level_error_page',
            $default
        );
    }

    /**
     * Get Cancelled Level URL
     *
     * @param  string $default Optional default value.
     * @return string
     */
    public function cancelled_url($default = '')
    {
        return $this->get_redirect_url(
            'membership_cancelled_type',
            'membership_cancelled_internal',
            'membership_cancelled',
            'membership_cancelled',
            'membership_cancelled_type_',
            'membership_cancelled_internal_',
            'membership_cancelled_',
            'membership_cancelled'
        );
    }

    /**
     * Get Expired URL
     *
     * @param  string $default Optional default URL.
     * @return string
     */
    public function expired_url($default = '')
    {
        return $this->get_redirect_url(
            'membership_expired_type',
            'membership_expired_internal',
            'membership_expired',
            'membership_expired',
            'membership_expired_type_',
            'membership_expired_internal_',
            'membership_expired_',
            'membership_expired',
            $default
        );
    }

    /**
     * Get For Approval URL
     *
     * @param  string $default Optional default URL.
     * @return string
     */
    public function for_approval_url($default = '')
    {
        return $this->get_redirect_url(
            'membership_forapproval_type',
            'membership_forapproval_internal',
            'membership_forapproval',
            'membership_forapproval',
            'membership_forapproval_type_',
            'membership_forapproval_internal_',
            'membership_forapproval_',
            'membership_forapproval',
            $default
        );
    }

    /**
     * Get For Confirmation URL
     *
     * @param  string $default Optional default value.
     * @return string
     */
    public function for_confirmation_url($default = '')
    {
        return $this->get_redirect_url(
            'membership_forconfirmation_type',
            'membership_forconfirmation_internal',
            'membership_forconfirmation',
            'membership_forconfirmation',
            'membership_forconfirmation_type_',
            'membership_forconfirmation_internal_',
            'membership_forconfirmation_',
            'membership_forconfirmation',
            $default
        );
    }

    /**
     * Get Duplicate POST URL
     *
     * @param  string $default Optional default value.
     * @return string
     */
    public function duplicate_post_url($default = '')
    {
        global $post;
        if (! $default) {
            $default = get_bloginfo('url');
        }

        $type = $this->get_option('duplicate_post_error_page_type');
        if (false === $type) {
            $url = $this->get_option('duplicate_post_error_page_internal');
            $url = $url ? get_permalink($url) : $this->get_option('duplicate_post_error_page');
        } else {
            if ('text' === $type) {
                $url = add_query_arg('sp', 'duplicate_post_error_page', $this->magic_page());
            } elseif ('internal' === $type) {
                $url = $this->get_option('duplicate_post_error_page_internal');
                $url = get_permalink($url);
            } else {
                $url = $this->get_option('duplicate_post_error_page');
            }
        }

        if (! $url) {
            $url = add_query_arg('sp', 'duplicate_post_error_page', $this->magic_page());
        }

        if (isset($post->ID) || $post->ID > 0) {
            $url2 = $this->get_option('duplicate_post_error_page_internal_' . $post->ID);
            $url2 = $url2 ? get_permalink($url2) : $this->get_option('duplicate_post_error_page_' . $post->ID);
            $url  = $url2 ? $url2 : $url;
        }

        $url = ! $url ? $default : $this->append_current_uri($url);
        return $url;
    }

    /**
     * Append the Current Request URI to the passed URL
     *
     * @param  string $url URL.
     * @return string
     */
    public function append_current_uri($url)
    {
        $qs   = ( false === strpos($url, '?') ) ? '?' : '&';
        $url .= $qs . 'wlfrom=' . rawurlencode(wlm_server_data()['REQUEST_URI']);
        return $url;
    }

    /**
     * Get Page IDs used for Specific System Pages
     *
     * @return array page IDs
     */
    public function get_specific_system_pages_id()
    {
        global $wpdb;
        $table    = $this->table_names->options;
        $page_ids = $wpdb->get_results(
            'SELECT option_name, option_value FROM `' . esc_sql($table) . '`'
            . 'WHERE (option_name LIKE "non_members_error_page_internal_%"'
            . ' OR option_name LIKE "wrong_level_error_page_internal_%"'
            . ' OR option_name LIKE "membership_cancelled_internal_%"'
            . ' OR option_name LIKE "membership_expired_internal%_"'
            . ' OR option_name LIKE "membership_forapproval_internal_%"'
            . ' OR option_name LIKE "membership_forconfirmation_internal_%")'
            . ' AND option_value > 0'
        );
        $result   = [];

        // We only want to include the post/page ID as a specific system page if the system page type is set to "Page".
        // Don't include it if the system page type is set to either Default, Message or URL.
        foreach ($page_ids as $page) {
            $opt_name = str_replace('internal', 'type', $page->option_name);
            $x_type_value = $this->get_option($opt_name);

            if ('page' === $x_type_value) {
                $result[] = $page->option_value;
            }
        }
        return $result;
    }

    /**
     * Creates a post from a template file.
     *
     * @param  string $template_name The name of the template.
     * @param  array  $postarr       Additional post/page variables.
     * @param  string $ext           The extension of the template. Default 'php'.
     * @return integer              Post ID.
     */
    public function create_post_from_template($template_name, $postarr = [], $ext = 'php')
    {
        $template_name = sprintf('%s/resources/page_templates/%s.%s', $this->plugin_dir, $template_name, $ext);
        include $template_name;
        $post['post_title']   = $content_title;
        $post['post_content'] = $content;
        $post                 = array_merge($post, $postarr);
        $id                   = wp_insert_post($post, true);
        return $id;
    }
}
