<?php

namespace WishListMember\Autoresponders;

class ArpReach
{
    /**
     * Triggered when a user registers.
     * Hook: wishlistmember_user_registered
     *
     * @param integer $user_id The ID of the registered user.
     * @param array   $data    An array containing user data, including 'wpm_id'.
     */
    public static function user_registered($user_id, $data)
    {
        self::added_to_level($user_id, [$data['wpm_id']]);
    }

    /**
     * Triggered when a user is added to a level/PPP.
     * Hooks: wishlistmember_add_user_levels_shutdown,
     * wishlistmember_confirm_user_levels
     * wishlistmember_approve_user_levels
     * wishlistmember_payperpost_added
     *
     * @param integer $user_id  The ID of the user.
     * @param mixed   $level_id The level ID(s) to which the user was added.
     */
    public static function added_to_level($user_id, $level_id)
    {
        if ('wishlistmember_payperpost_added' === current_action()) {
            $level_id = array_map(
                function ($level) {
                    return 'payperpost-' . $level;
                },
                (array) $level_id
            );
        } else {
            $level_id = wlm_remove_inactive_levels($user_id, $level_id);
        }

        self::process($user_id, $level_id, 'added');
    }

    /**
     * Triggered when a user is removed from a level/PPP.
     *
     * @param integer $user_id  The ID of the user.
     * @param mixed   $level_id The level ID(s) from which the user was removed.
     */
    public static function removed_from_level($user_id, $level_id)
    {
        self::process($user_id, $level_id, 'removed');
    }

    /**
     * Triggered when a user is uncanelled from a level/PPP.
     *
     * @param integer $user_id The ID of the user.
     * @param mixed   $levels  The level ID(s) from which the user was uncancelled.
     */
    public static function uncancelled_from_level($user_id, $levels)
    {
        self::process($user_id, $levels, 'uncancelled');
    }

    /**
     * Triggered when a user is cancelled from a level/PPP.
     *
     * @param integer $user_id The ID of the user.
     * @param mixed   $levels  The level ID(s) from which the user was cancelled.
     */
    public static function cancelled_from_level($user_id, $levels)
    {
        self::process($user_id, $levels, 'cancelled');
    }

    /**
     * Processes the subscription|unsubscription of a user based on the specified action and levels
     *
     * @param mixed  $email_or_id The user's email address or ID.
     * @param mixed  $levels      The level ID(s) to process.
     * @param string $action      The action to perform ('added', 'removed', 'uncancelled', 'cancelled').
     */
    public static function process($email_or_id, $levels, $action)
    {
        static $ar;

        if (empty($ar)) {
            $ar = ( new \WishListMember\Autoresponder('arpreach') )->settings;
        }

        // Get email address.
        if (is_numeric($email_or_id)) {
            $userdata = get_userdata($email_or_id);
            if (! $userdata) {
                return; // Invalid user_id.
            }
        } elseif (filter_var($email_or_id, FILTER_VALIDATE_EMAIL)) {
            $userdata = get_user_by('email', $email_or_id);
            if (! $userdata) {
                return; // Invalid user_id.
            }
        } else {
            return; // Email_or_id is neither a valid ID or email address.
        }
        $email = $userdata->user_email;
        $fname = $userdata->first_name;
        $lname = $userdata->last_name;

        // Make sure email is not temp.
        if (! wlm_trim($email) || preg_match('/^temp_[0-9a-f]+/i', $email)) {
            return;
        }

        // Make sure levels is an array.
        if (! is_array($levels)) {
            $levels = [$levels];
        }

        foreach ($levels as $level_id) {
            $add    = wlm_or($ar['list_actions'][ $level_id ][ $action ]['add'], '');
            $remove = wlm_or($ar['list_actions'][ $level_id ][ $action ]['remove'], '');
            if ($add) {
                self::send_data($add, $email, $fname, $lname, true);
            }
            if ($remove) {
                self::send_data($remove, $email, $fname, $lname, false);
            }
        }
    }

    /**
     * Sends data to ArpReach for subscribing or unsubscribing a user.
     *
     * @param string  $post_url  The URL to send the data to.
     * @param string  $email     The email address of the user.
     * @param string  $fname     The first name of the user.
     * @param string  $lname     The last name of the user.
     * @param boolean $subscribe Indicates whether to subscribe (true) or unsubscribe (false) the user.
     */
    public static function send_data($post_url, $email, $fname, $lname, $subscribe)
    {

        if ($post_url) {
            $http_agent = 'WLM_ARPREACH_AGENT';
            $post_data  = [
                'email_address' => $email,
                'first_name'    => $fname,
                'last_name'     => $lname,
            ];
            if ($subscribe) {
                $post_data['unsubscribe'] = 0;
            } else {
                $post_data['unsubscribe'] = 1;
            }

            if ($subscribe || isset($post_data['unsubscribe']) && ! $subscribe) {
                wp_remote_post(
                    $post_url,
                    [
                        'blocking'   => false,
                        'user-agent' => $http_agent,
                        'body'       => $post_data,
                    ]
                );
            }
        }
    }
}
