<?php

namespace WishListMember\Infusionsoft;

class WLM_Infusionsoft_Connection
{
    /**
     * Infusionsoft services
     *
     * @var \WishListMember\Infusionsoft\Infusionsoft
     */
    private $services = false;

    /**
     * Infusionsoft encryption key
     *
     * @var string
     */
    private $enckey = '';

    /**
     * Reconnect flag
     *
     * @var boolean
     */
    private $recon = false;

    /**
     * Logging flag
     *
     * @var boolean
     */
    private $enable_log = false;

    /**
     * Infusionsoft suffix
     *
     * @var string
     */
    private $token_suffix = '';

    /**
     * Keap integration connection URL
     *
     * @var string
     */
    private static $connection_url = 'https://keap-auth.wishlistmember.com';

    /**
     * Integration type. Either 'payment' or 'autoresponder'
     *
     * @var string
     */
    private $integration_type = 'payment';

    /**
     * Keap integration app client ID
     *
     * @var string
     */
    private $client_id = '';

    public function __construct($integration_type = 'payment', $client_id = '')
    {
        $this->client_id = $client_id;
        $this->integration_type = $integration_type;
        $this->token_suffix = '_' . $this->integration_type;
    }

    public function get_connection()
    {
        if (false === $this->services || $this->recon) {
            // If API Key is not set, we are using OAuth.
            if (empty($this->enckey)) {
                $token_refresh = get_transient('keap_token_refresh' . $this->token_suffix);
                $oauth_token = wishlistmember_instance()->get_option('keap_oauth_token' . $this->token_suffix);
                $this->services = self::get_oauth_object();
                if ($this->services && $oauth_token) {
                    $this->services->setToken($oauth_token);
                    $this->recon = false;
                    // We need to refresh the token sincethe marker expired.
                    if (! $token_refresh || $this->services->isTokenExpired()) {
                        try {
                            $response = json_decode(
                                wp_remote_retrieve_body(
                                    wp_remote_post(
                                        self::$connection_url,
                                        [
                                            'body' => [
                                                'r' => $this->services->getRedirectUri(),
                                                'a' => $this->integration_type,
                                                'refresh' => $oauth_token->getRefreshToken(),
                                            ],
                                        ]
                                    )
                                )
                            );

                            if (!is_object($response)) {
                                return $this->log_and_reconnect('Invalid Response when refreshing token');
                            }

                            if (isset($response->error)) {
                                return $this->log_and_reconnect('Response Error: ' . $response->error);
                            }

                            $newtoken = new \WishListMember\Infusionsoft\Token();
                            $newtoken->setAccessToken($response->accessToken);
                            $newtoken->setRefreshToken($response->refreshToken);
                            $newtoken->setEndOfLife($response->endOfLife);
                            $newtoken->setExtraInfo((array)$response->extraInfo);

                            if ($newtoken->isExpired()) {
                                return $this->log_and_reconnect('Token Expired, needs to reconnect to Keap');
                            }

                            wishlistmember_instance()->save_option('keap_oauth_token' . $this->token_suffix, $newtoken);
                            /**
                            * We'll create a transient for 23 hours.
                            * This acts as marker for us when the token needs to be refreshed.
                            * This is to prevent the token from being refreshed too often.
                            */
                            set_transient('keap_token_refresh' . $this->token_suffix, $newtoken, 60 * 60 * 23);
                            $this->services->setToken($newtoken);
                        } catch (\Exception $e) {
                            error_log($e->getMessage());
                            $this->services = false;
                            $this->recon = true;
                        }
                    }
                } else {
                    $this->services = false;
                    $this->recon = true;
                }
            } else {
                $config = ['apikey' => $this->enckey];
                if ($this->enable_log) {
                    $config['debug'] = true;
                }
                try {
                    $this->services = new \WishListMember\Infusionsoft\Infusionsoft($config);
                    $this->recon = false;
                } catch (\Exception $e) {
                    error_log($e->getMessage());
                    $this->services = false;
                    $this->recon = true;
                }
            }
        }
        return $this->services;
    }

    /**
     * Log and force reconnect
     *
     * @param string $error The error message.
     */
    private function log_and_reconnect($error)
    {
        error_log('WLM Keap Integration: ' . $error);
        $this->services = false;
        $this->recon    = true;
        return $this->services;
    }

    public function set_connection($enckey)
    {
        $this->enckey = $enckey;
        $this->recon = true;
    }

    public function enable_logging()
    {
        $this->enable_log = true;
    }

    private function get_redirect_uri()
    {
        $action_type = 'wlm_payment_oauth';
        if ('autoresponder' === $this->integration_type) {
            $action_type = 'wlm_autoresponder_oauth';
        }
        return admin_url('admin.php') . '?action=' . $action_type;
    }

    public function get_oauth_object()
    {
        $params = [
            'clientId' => $this->client_id,
            'clientSecret' => '',
            'redirectUri' => $this->get_redirect_uri(),
        ];
        $ifsdk_obj = false;
        try {
            $ifsdk_obj = new \WishListMember\Infusionsoft\Infusionsoft($params);
        } catch (\Exception $e) {
            error_log($e->getMessage());
        }
        return $ifsdk_obj;
    }

    public function get_token_from_code($code)
    {
        $response = json_decode(
            wp_remote_retrieve_body(
                wp_remote_post(
                    self::$connection_url,
                    [
                        'body' => [
                            'r' => $this->get_redirect_uri(),
                            'a' => $this->integration_type,
                            'code' => $code,
                        ],
                    ]
                )
            )
        );

        if (!is_object($response)) {
            return false;
        }

        $token = new \WishListMember\Infusionsoft\Token();
        $token->setAccessToken($response->accessToken);
        $token->setRefreshToken($response->refreshToken);
        $token->setEndOfLife($response->endOfLife);
        $token->setExtraInfo((array)$response->extraInfo);

        return $token->isExpired() ? false : $token;
    }

    public function get_oauth_authorize_url()
    {
        $response = json_decode(
            wp_remote_retrieve_body(
                wp_remote_post(
                    self::$connection_url,
                    [
                        'body' => [
                            'r' => $this->get_redirect_uri(),
                            'a' => $this->integration_type,
                        ],
                    ]
                )
            )
        );
        return is_object($response) ? ($response->url ?? '') : '';
    }
}
